
seasonLength = 1; % 125; % season length in days (data length is 126 days, 112 if you need co2 injection)
firstDay = 26.125; %15*ones(1,numSim); % days since beginning of data        

%% LED simulations
[outdoor, indoor, controls, startTime] = loadGreenhouseData(firstDay, seasonLength, 'led', 'none');

indoor(:,3) = vaporDens2pres(indoor(:,2), indoor(:,3));
        % convert vapor density to vapor pressure

indoor(:,4) = 1e6*co2ppm2dens(indoor(:,2),indoor(:,4));
        % convert co2 from ppm to mg m^{-3}
 
secsInYear = seconds(startTime-datetime(year(startTime),1,1,0,0,0));
    % number of seconds since beginning of year to startTime

outdoor(:,7) = skyTempRdam(outdoor(:,3), datenum(startTime)+outdoor(:,1)/86400); % add sky temperature
outdoor(:,8) = soilTempNl(secsInYear+outdoor(:,1)); % add soil temperature

led = createBvModel(outdoor, startTime, controls, indoor);
setLedParams(led);

setParam(led,'cLeakage', 3e-5);
setParam(led,'aRoof', 52.2);

setParam(led,'kBlScr',5e-4);

setParam(led,'kThScr',5e-4);

setParam(led, 'cWgh', 0.02); % default is 0.09
setParam(led, 'cDgh', 0.35); % default is 0.75

led.x.cLeaf.val = 0.7*6240*10;
led.x.cStem.val = 0.25*6240*10;
led.x.cFruit.val = 0.05*6240*10;

setParam(led, 'cHecIn', 3.5);

ledNoCool = StateSpaceModel(led);
setParam(ledNoCool,'lampCool',0);

%% HPS simulations
[outdoor, indoor, controls, startTime] = loadGreenhouseData(firstDay, seasonLength, 'hps', 'none');

indoor(:,3) = vaporDens2pres(indoor(:,2), indoor(:,3));
        % convert vapor density to vapor pressure

indoor(:,4) = 1e6*co2ppm2dens(indoor(:,2),indoor(:,4));
        % convert co2 from ppm to mg m^{-3}
 
secsInYear = seconds(startTime-datetime(year(startTime),1,1,0,0,0));
    % number of seconds since beginning of year to startTime

outdoor(:,7) = skyTempRdam(outdoor(:,3), datenum(startTime)+outdoor(:,1)/86400); % add sky temperature
outdoor(:,8) = soilTempNl(secsInYear+outdoor(:,1)); % add soil temperature

hps = createBvModel(outdoor, startTime, controls, indoor);
setHpsParams(hps);

setParam(hps,'cLeakage', 3e-5);
setParam(hps,'aRoof', 52.2);

setParam(hps,'kBlScr',5e-4);

setParam(hps,'kThScr',5e-4);

setParam(hps, 'cWgh', 0.02); % default is 0.09
setParam(hps, 'cDgh', 0.35); % default is 0.75

hps.x.cLeaf.val = 0.7*6240*10;
hps.x.cStem.val = 0.25*6240*10;
hps.x.cFruit.val = 0.05*6240*10;


%% run the simulations
seasonLength = led.t.val(2)/86400; % season length in days, after filtering

timer = tic;
fprintf('Start time: %s; Estimated run time: %s; ETA: %s\n', ...
    datetime('now'), datestr(seconds(3*seasonLength*90), 'HH:MM:SS'), ...
    datestr(datetime('now')+seconds(3*seasonLength*90)));

options = [];

odeSolve(led,'ode15s', options);
odeSolve(ledNoCool,'ode15s', options);
odeSolve(hps,'ode15s', options);

fprintf('Actual run time: %s; End time: %s\n', ...
     datestr(seconds(toc(timer)), 'HH:MM:SS'), datetime('now'));

led = changeRes(led,300);
hps = changeRes(hps,300);
ledNoCool = changeRes(ledNoCool,300);

%% plot
figure;
subplot(3,1,1)
plot(hps.x.tAir)
hold on
plot(hps.x.tLamp)
ylabel('Temperature (C)')
axis([0 9e4 0 250])
yyaxis right
plot(hps.a.lampIn)
ylabel('Lamp input (W m^{-2})')
legend('Air temperature (C)','Lamp temperature (C)','Lamp input (W m^{-2})')
xticks(2700+0:3600:9e4);
xtickNumToDate(hps,'HH:00');
xtickangle(45);

labels = xticklabels;
for k=2:2:length(labels)
    labels{k} = '';
end
xticklabels(labels);

title('HPS temperature');
grid

subplot(3,1,2)
plot(led.x.tAir)
hold on
plot(led.x.tLamp)
ylabel('Temperature (C)')
yyaxis right
plot(led.a.lampIn)
ylabel('Lamp input (W m^{-2})')
xticks(2700+0:3600:9e4);
xtickNumToDate(hps,'HH:00');
xtickangle(45);

labels = xticklabels;
for k=2:2:length(labels)
    labels{k} = '';
end
xticklabels(labels);
title('LED temperature (with cooling)')
grid

subplot(3,1,3)
plot(ledNoCool.x.tAir)
hold on
plot(ledNoCool.x.tLamp)
ylabel('Temperature (C)')
yyaxis right
plot(ledNoCool.a.lampIn)
ylabel('Lamp input (W m^{-2})')
xticks(2700+0:3600:9e4);
xtickNumToDate(hps,'HH:00');
xtickangle(45);

labels = xticklabels;
for k=2:2:length(labels)
    labels{k} = '';
end
xticklabels(labels);
title('LED temperature (no cooling)')
xlabel('Time of day (hours)')
grid
